package com.tencent.start.cgs.tools;

import java.io.DataOutput;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteOrder;
import java.util.Arrays;

@SuppressWarnings("unused")
public class RandomAccessByteArrayOutputStream extends OutputStream implements DataOutput {
    protected boolean bigEndian = true;
    protected boolean nativeByteOrder = (bigEndian == (ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN));

    public final ByteOrder order() {
        return bigEndian ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN;
    }

    public final RandomAccessByteArrayOutputStream order(ByteOrder bo) {
        bigEndian = (bo == ByteOrder.BIG_ENDIAN);
        nativeByteOrder = (bigEndian == (ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN));
        return this;
    }

    protected byte[] buf;
    protected int pos;
    protected int count;

    public RandomAccessByteArrayOutputStream() {
        this(32);
    }

    public RandomAccessByteArrayOutputStream(int size) {
        if (size < 0) {
            throw new IllegalArgumentException("Negative initial size: " + size);
        }
        buf = new byte[size];
    }

    private void ensureCapacity(int minCapacity) {
        // overflow-conscious code
        if (minCapacity - buf.length > 0) {
            grow(minCapacity);
        }
    }

    private static final int MAX_ARRAY_SIZE = Integer.MAX_VALUE - 8;

    private void grow(int minCapacity) {
        // overflow-conscious code
        int newCapacity = buf.length;
        int cap;
        while (newCapacity < minCapacity && (cap = newCapacity << 1) < MAX_ARRAY_SIZE) {
            newCapacity = cap;
        }
        if (newCapacity < minCapacity) {
            newCapacity = minCapacity;
        }
        if (newCapacity - MAX_ARRAY_SIZE > 0) {
            newCapacity = hugeCapacity(minCapacity);
        }
        if (newCapacity > buf.length) {
            buf = Arrays.copyOf(buf, newCapacity);
        }
    }

    private static int hugeCapacity(int minCapacity) {
        if (minCapacity < 0) {
            // overflow
            throw new OutOfMemoryError();
        }
        return (minCapacity > MAX_ARRAY_SIZE) ? Integer.MAX_VALUE : MAX_ARRAY_SIZE;
    }

    @Override
    public void write(int b) {
        ensureCapacity(pos + 1);
        buf[pos] = (byte) b;
        pos += 1;
        if (pos > count) {
            count = pos;
        }
    }

    @SuppressWarnings("NullableProblems")
    @Override
    public void write(byte[] b, int off, int len) {
        if ((off < 0) || (off > b.length) || (len < 0) || ((off + len) - b.length > 0)) {
            throw new IndexOutOfBoundsException();
        }
        ensureCapacity(pos + len);
        System.arraycopy(b, off, buf, pos, len);
        pos += len;
        if (pos > count) {
            count = pos;
        }
    }

    public void writeTo(OutputStream out) throws IOException {
        out.write(buf, 0, count);
    }

    public void reset() {
        pos = 0;
        count = 0;
    }

    public byte[] toByteArray() {
        return Arrays.copyOf(buf, count);
    }

    public String toString() {
        return new String(buf, 0, count);
    }

    public void seek(int position) {
        if (position > count) {
            grow(position);
        }
        pos = position;
    }

    public int getPosition() {
        return pos;
    }

    public void setLength(int newCount) {
        if (newCount > count) {
            grow(newCount);
        }
        count = newCount;
    }

    public int getLength() {
        return count;
    }

    @Override
    public void writeBoolean(boolean b) {
        this.write(b ? 1 : 0);
    }

    @Override
    public void writeBytes(String s) {
        byte[] data = s.getBytes();
        this.write(data, 0, data.length);
    }

    @SuppressWarnings("all")
    @Override
    public void writeChars(String s) {
        int length = s.length();
        int length2 = 2 * length;
        byte[] b1 = new byte[length2];
        char[] b2 = new char[length];
        s.getChars(0, length, b2, 0);
        int var6 = 0;

        for(int i = 0; var6 < length; ++var6) {
            b1[i++] = (byte)(b2[var6] >>> 8);
            b1[i++] = (byte)(b2[var6] >>> 0);
        }
        this.write(b1, 0, length2);
    }

    @SuppressWarnings("all")
    @Override
    public void writeUTF(String str) throws IOException {
        RandomAccessDataOutput.writeUTF(str, this);
    }

    @Override
    public void writeByte(int v) {
        this.write(v);
    }

    @SuppressWarnings("all")
    @Override
    public void writeShort(int v) {
        if (nativeByteOrder) {
            this.write(v >>> 0 & 255);
            this.write(v >>> 8 & 255);
        } else {
            this.write(v >>> 8 & 255);
            this.write(v >>> 0 & 255);
        }
    }

    @SuppressWarnings("all")
    @Override
    public void writeChar(int v) {
        if (nativeByteOrder) {
            this.write(v >>> 0 & 255);
            this.write(v >>> 8 & 255);
        } else {
            this.write(v >>> 8 & 255);
            this.write(v >>> 0 & 255);
        }
    }

    @SuppressWarnings("all")
    @Override
    public void writeInt(int v) {
        if (nativeByteOrder) {
            this.write(v >>>  0 & 255);
            this.write(v >>>  8 & 255);
            this.write(v >>> 16 & 255);
            this.write(v >>> 24 & 255);
        } else {
            this.write(v >>> 24 & 255);
            this.write(v >>> 16 & 255);
            this.write(v >>> 8 & 255);
            this.write(v >>> 0 & 255);
        }
    }

    @SuppressWarnings("all")
    @Override
    public void writeLong(long v) {
        if (nativeByteOrder) {
            this.write((int) (v >>>  0) & 255);
            this.write((int) (v >>>  8) & 255);
            this.write((int) (v >>> 16) & 255);
            this.write((int) (v >>> 24) & 255);
            this.write((int) (v >>> 32) & 255);
            this.write((int) (v >>> 40) & 255);
            this.write((int) (v >>> 48) & 255);
            this.write((int) (v >>> 56) & 255);
        } else {
            this.write((int) (v >>> 56) & 255);
            this.write((int) (v >>> 48) & 255);
            this.write((int) (v >>> 40) & 255);
            this.write((int) (v >>> 32) & 255);
            this.write((int) (v >>> 24) & 255);
            this.write((int) (v >>> 16) & 255);
            this.write((int) (v >>> 8) & 255);
            this.write((int) (v >>> 0) & 255);
        }
    }

    @Override
    public void writeFloat(float v) {
        this.writeInt(Float.floatToIntBits(v));
    }

    @Override
    public void writeDouble(double v) {
        this.writeLong(Double.doubleToLongBits(v));
    }
}
